

flat varying vec3 ambientUp;
flat varying vec3 ambientDown;
flat varying vec3 ambientLeft;
flat varying vec3 ambientRight;
flat varying vec3 ambientB;
flat varying vec3 ambientF;
flat varying float ambientA;
flat varying vec4 sunColor;
varying vec2 texcoord;

#include "/lib/sky_function.glsl"

vec3 GenerateUnitVector(vec2 xy)
{
    const float TAU = 6.28318530718; // 2*pi
    xy.x *= TAU;
    xy.y = xy.y * 2.0 - 1.0;
    float r = sqrt(1.0 - xy.y * xy.y);
    return vec3(vec2(sin(xy.x), cos(xy.x)) * r, xy.y);
}

void calculateLighting2(
    out vec3 ambientUp,
    out vec3 ambientLeft,
    out vec3 ambientRight,
    out vec3 ambientB,
    out vec3 ambientF,
    out vec3 ambientDown,
    out float ambientA,
    out vec4 sunColor)
{
    // Precompute sun visibility.
    float clampedSunElevation = clamp(sunElevation, 0.0, 0.05);
    float t = clampedSunElevation / 0.05;
    float sunVis = t * t;
    sunColor = vec4(sunLight, sunVis);

    // Sample grid setup.
    const ivec2 samples = ivec2(32, 16);
    const vec2 invSamples = vec2(1.0 / float(samples.x), 1.0 / float(samples.y));
    const float sampleWeight = 2.0 / float(samples.x * samples.y);

    // Initialize ambient accumulators.
    ambientRight = ambientUp = ambientF = vec3(0.0);
    ambientLeft = ambientDown = ambientB = vec3(0.0);

    // Loop over the sample grid.
    for (int x = 0; x < samples.x; ++x)
    {
        for (int y = 0; y < samples.y; ++y)
        {
            vec2 uv = (vec2(x, y) + 0.5) * invSamples;
            vec3 dir = GenerateUnitVector(uv);
            dir.y = abs(dir.y); // enforce upward direction

            vec3 skySample = skyFromTex2(dir, colortex6).rgb;

            // Precompute directional weights.
            float posX = max(dir.x, 0.0);
            float posY = max(dir.y, 0.0);
            float posZ = max(dir.z, 0.0);
            float negX = max(-dir.x, 0.0);
            float negZ = max(-dir.z, 0.0);

            ambientRight += skySample * posX;
            ambientUp += skySample * posY;
            ambientF += skySample * posZ;
            ambientLeft += skySample * negX;
            ambientB += skySample * negZ;
        }
    }

    // Apply sample weight.
    ambientRight *= sampleWeight;
    ambientUp *= sampleWeight;
    ambientF *= sampleWeight;
    ambientLeft *= sampleWeight;
    ambientB *= sampleWeight;
#ifdef OVERWORLD
    // Fake skylight bounce.
    const float fakeBounceAlbedo = 0.5;
    ambientRight += ambientUp * fakeBounceAlbedo * 0.5;
    ambientF += ambientUp * fakeBounceAlbedo * 0.5;
    ambientLeft += ambientUp * fakeBounceAlbedo * 0.5;
    ambientDown += ambientUp * fakeBounceAlbedo;
    ambientB += ambientUp * fakeBounceAlbedo * 0.5;

    // Compute light direction.
    float lightDir = (sunVis >= 1e-5) ? 1.0 : -1.0;

    vec3 sunLightLocal = sunLight;

    vec3 bouncedSun = sunLightLocal * 0.001;

    // Precompute bounce clamping values.
    float clampUp = clamp(-lightDir * sunVec.y + 3.0, 0.0, 4.0);
    float clampLeft = clamp(lightDir * sunVec.x + 3.0, 0.0, 4.0);
    float clampRight = clamp(-lightDir * sunVec.x + 3.0, 0.0, 4.0);
    float clampBack = clamp(-lightDir * sunVec.z + 3.0, 0.0, 4.0);
    float clampFront = clamp(lightDir * sunVec.z + 3.0, 0.0, 4.0);
    float clampDown = clamp(lightDir * sunVec.y + 3.0, 0.0, 4.0);

    ambientUp += bouncedSun * clampUp;
    ambientLeft += bouncedSun * clampLeft;
    ambientRight += bouncedSun * clampRight;
    ambientB += bouncedSun * clampBack;
    ambientF += bouncedSun * clampFront;
    ambientDown += bouncedSun * clampDown * 0.7;
#endif
    // Compute ambient overall luminance.
    ambientA = luma((ambientUp + ambientLeft + ambientRight + ambientB + ambientF + ambientDown) / 6.0);
}
void calculateLighting3(
    out vec3 ambientUp,
    out vec3 ambientLeft,
    out vec3 ambientRight,
    out vec3 ambientB,
    out vec3 ambientF,
    out vec3 ambientDown,
    out float ambientA,
    out vec4 sunColor)
{
    // Sun visibility
    float clampedSunElevation = clamp(sunElevation, 0.0, 0.05);
    float t = clampedSunElevation / 0.05;
    float sunVis = t * t;
    sunColor = vec4(sunLight, sunVis);

    // Sampling config
    const ivec2 samples = ivec2(32, 16);
    const vec2 invSamples = vec2(1.0 / float(samples.x), 1.0 / float(samples.y));
    const float sampleWeight = 2.0 / float(samples.x * samples.y);

    // Initialize ambient directions
    ambientRight = ambientLeft = ambientF = ambientB = ambientUp = ambientDown = vec3(0.0);

    // Full sky sampling (both hemispheres)
    for (int x = 0; x < samples.x; ++x)
    {
        for (int y = 0; y < samples.y; ++y)
        {
            vec2 uv = (vec2(x, y) + 0.5) * invSamples;
            vec3 dir = GenerateUnitVector(uv); // maps uv ∈ [0,1]^2 to unit sphere

            vec3 skySample = skyFromTex2(dir, colortex6).rgb;

            // Weighted direction contributions
            float posX = max(dir.x, 0.0);
            float negX = max(-dir.x, 0.0);
            float posY = max(dir.y, 0.0);
            float negY = max(-dir.y, 0.0);
            float posZ = max(dir.z, 0.0);
            float negZ = max(-dir.z, 0.0);

            ambientRight += skySample * posX;
            ambientLeft += skySample * negX;
            ambientUp += skySample * posY;
            ambientDown += skySample * negY;
            ambientF += skySample * posZ;
            ambientB += skySample * negZ;
        }
    }

    // Normalize
    ambientRight *= sampleWeight;
    ambientLeft *= sampleWeight;
    ambientF *= sampleWeight;
    ambientB *= sampleWeight;
    ambientUp *= sampleWeight;
    ambientDown *= sampleWeight * 0.5; // simulate terrain occlusion

    // Add direct sun bounce into ambientUp
    vec3 bouncedSun = sunLight * sunVis * 0.05;
    ambientUp += bouncedSun;

    // Multi-bounce simulation with directionally-weighted redistribution
    const float bounceStrengths[3] = float[](0.5, 0.3, 0.2);
    for (int i = 0; i < 3; ++i)
    {
        vec3 dirs[6] = vec3[](ambientRight, ambientLeft, ambientF, ambientB, ambientUp, ambientDown);
        vec3 next[6] = vec3[](vec3(0.0), vec3(0.0), vec3(0.0), vec3(0.0), vec3(0.0), vec3(0.0));
        float b = bounceStrengths[i];

        // Simple redistribution model: bounce energy from each direction into others
        for (int d = 0; d < 6; ++d)
        {
            vec3 bounced = dirs[d] * b;
            // Directions: 0=R, 1=L, 2=F, 3=B, 4=U, 5=D

            if (d == 0 || d == 1)
            {                              // Right or Left
                next[0] += bounced * 0.20; // Right
                next[1] += bounced * 0.20; // Left
                next[2] += bounced * 0.15; // Fwd
                next[3] += bounced * 0.15; // Back
                next[4] += bounced * 0.15; // Up
                next[5] += bounced * 0.15; // Down
            }
            else if (d == 2 || d == 3)
            { // Fwd or Back
                next[0] += bounced * 0.15;
                next[1] += bounced * 0.15;
                next[2] += bounced * 0.20;
                next[3] += bounced * 0.20;
                next[4] += bounced * 0.15;
                next[5] += bounced * 0.15;
            }
            else if (d == 4)
            { // Up
                next[0] += bounced * 0.10;
                next[1] += bounced * 0.10;
                next[2] += bounced * 0.10;
                next[3] += bounced * 0.10;
                next[4] += bounced * 0.10;
                next[5] += bounced * 0.50; // Light bounces down
            }
            else if (d == 5)
            { // Down
                next[0] += bounced * 0.10;
                next[1] += bounced * 0.10;
                next[2] += bounced * 0.10;
                next[3] += bounced * 0.10;
                next[4] += bounced * 0.50; // Light bounces up
                next[5] += bounced * 0.10;
            }
        }

        // Accumulate and renormalize
        for (int d = 0; d < 6; ++d)
        {
            dirs[d] += next[d];
            dirs[d] *= 1.0 / (1.0 + b); // optional: approximate energy conservation
        }

        ambientRight = dirs[0];
        ambientLeft = dirs[1];
        ambientF = dirs[2];
        ambientB = dirs[3];
        ambientUp = dirs[4];
        ambientDown = dirs[5];
    }

    // Final ambient luminance average
    ambientA = luma((ambientUp + ambientDown + ambientLeft + ambientRight + ambientF + ambientB) / 6.0);
}

void main()
{

    texcoord = gl_MultiTexCoord0.xy;

    calculateLighting2(ambientUp, ambientLeft, ambientRight, ambientB, ambientF, ambientDown, ambientA, sunColor);

    gl_Position = ftransform();
}
